package gov.va.med.mhv.usermgmt.persist.hibernate;

import gov.va.med.mhv.usermgmt.bizobj.PatientInformationBO;
import gov.va.med.mhv.usermgmt.enumeration.AuthenticationStatus;
import gov.va.med.mhv.usermgmt.enumeration.PatientCorrelationStatus;
import gov.va.med.mhv.usermgmt.persist.PatientInformationQueryDao;
import gov.va.med.mhv.usermgmt.transfer.InPersonAuthenticationCriteria;
import gov.va.med.mhv.usermgmt.transfer.PatientInformation;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.transform.AliasToBeanResultTransformer;
import org.tigris.atlas.persist.hibernate.DaoImpl;

public class PatientInformationQueryDaoImpl extends DaoImpl 
    implements PatientInformationQueryDao 
{

	private static final Log LOG = LogFactory.getLog(
        PatientInformationQueryDaoImpl.class);

	//MHV_CodeCR1387 - Added middle name and gender as part of query.
	//MHV_CodeCR1514 - US12.4 MVI Compliance Implementation - Added Correlation Status, Correlation Date, Matched DAte as part of query. -->
	private static final String [] PROPERTIES = new String [] {
        "firstName", "middleName", "lastName", "gender", "ssn", "email", "mhvId", "birthDate", 
        "authenticationStatus", "authenticationDate", 
        "authenticatingFacilityInfoName", "patientId", "username", "correlationStatus", "correlatedDate", "matchedDate"};
	private static final String BASE = 
        "select distinct patient.firstName, patient.middleName, patient.lastName, patient.gender, patient.ssn, " +
        "patient.email, patient.mhvId, patient.birthDate, " +
        "patient.authenticationStatus, patient.authenticationDate, " + 
        "patient.authenticatingFacilityInfoName, patient.patientId, " + 
        "patient.username, patient.correlationStatus, patient.correlatedDate, patient.matchedDate  from PatientInformationBO as patient";
	private static final String WHERE = " where ";
	private static final String AND = " and ";
	
	protected String getEntityName() {
		return PatientInformation.ENTITY;
	}

	public Collection findPatients(InPersonAuthenticationCriteria criteria) {
		processCriteria(criteria);
		Query query = getQuery(criteria);
		List results = query.list();
		
		//MHV_CodeCR1387 - Added debug messages
		if (LOG.isDebugEnabled()) {
			if(results != null) {
				LOG.debug("************** In PatientInformationQueryDaoImpl.findPatients() results: " + results.size());
				if(results.size()>0) {
					LOG.debug("************** In PatientInformationQueryDaoImpl.findPatients() results.get(0).getClass(): " + results.get(0).getClass());
				}
			}
		}
		return transformData(results);
	}
	
	/**
	 * Ensures that all boolean fields are at least set so we can use
	 * 'primitive' methods in if statement later.
	 *
	 * @param criteria The criteria to populate
	 */
	private void processCriteria(InPersonAuthenticationCriteria criteria) {
		if (criteria.getStatusAuthenticated() == null) {
			criteria.setStatusAuthenticated(Boolean.FALSE);
		}
		if (criteria.getStatusInProcess() == null) {
			criteria.setStatusInProcess(Boolean.FALSE);
		}
        if (criteria.getStatusPendingDataMismatch() == null) {
            criteria.setStatusPendingDataMismatch(Boolean.FALSE);
        }
		if (criteria.getStatusPrerequisitesComplete() == null) {
			criteria.setStatusPrerequisitesComplete(Boolean.FALSE);
		}
		if (criteria.getStatusUnauthenticated() == null) {
			criteria.setStatusUnauthenticated(Boolean.FALSE);
		}
		
		//MHV_CodeCR1514 - US12.4 MVI Compliance Implementation - Added Correlation Status checks -->
		if (criteria.getStatusMatched() == null) {
			criteria.setStatusMatched(Boolean.FALSE);
		}
		if (criteria.getStatusUnmatched() == null) {
			criteria.setStatusUnmatched(Boolean.FALSE);
		}
		if (criteria.getStatusCorrelated() == null) {
			criteria.setStatusCorrelated(Boolean.FALSE);
		}
		if (criteria.getStatusPendingCorrelation() == null) {
			criteria.setStatusPendingCorrelation(Boolean.FALSE);
		}
		if (criteria.getStatusPendingUncorrelated() == null) {
			criteria.setStatusPendingUncorrelated(Boolean.FALSE);
		}
		if (criteria.getStatusUncorrelated() == null) {
			criteria.setStatusUncorrelated(Boolean.FALSE);
		}
		if (criteria.getStatusFailedCorrelation() == null) {
			criteria.setStatusFailedCorrelation(Boolean.FALSE);
		}
		if (criteria.getStatusFailedUncorrelation() == null) {
			criteria.setStatusFailedUncorrelation(Boolean.FALSE);
		}
		if (criteria.getStatusCorrelationInvalid() == null) {
			criteria.setStatusCorrelationInvalid(Boolean.FALSE);
		}
	}

	private Collection transformData(List results) {
		AliasToBeanResultTransformer transformer = 
            new AliasToBeanResultTransformer(PatientInformationBO.class);
		Collection<PatientInformationBO> bos = 
            new ArrayList<PatientInformationBO>();
		for (Iterator i = results.iterator(); i.hasNext();) {
			Object [] obj = (Object []) i.next();
			PatientInformationBO bo = (PatientInformationBO) transformer.
                transformTuple(obj, PROPERTIES);
			bos.add(bo);
		}
		return bos;
	}

	private Query getQuery(InPersonAuthenticationCriteria criteria) {
		boolean caseSensitive = false;
		
		List<String> terms = new ArrayList<String>();
		Map<String, Object> values = new HashMap<String, Object>();
		// First letter of last name + last 4 of SSN
		if (!StringUtils.isBlank(criteria.getFirstPlusFour())) {
			String firstPlusFour = criteria.getFirstPlusFour();
			String first = firstPlusFour.substring(0, 1);
			String ssn = firstPlusFour.substring(1);
			String value = first+"*";
			terms.add(getPossiblyWildcardedTerm("patient.lastName", 
                "firstPlusFourName", value, caseSensitive));
			values.put("firstPlusFourName", getValue(value, caseSensitive));
			terms.add(getLikeTerm("patient.ssn", "firstPlusFourSsn"));
			values.put("firstPlusFourSsn", getValue("*"+ssn, caseSensitive));
		}
		
		// First name
		if (!StringUtils.isBlank(criteria.getFirstName())) {
			String value = criteria.getFirstName();
			terms.add(getPossiblyWildcardedTerm("patient.firstName", 
                "firstName", value, caseSensitive));
			values.put("firstName", getValue(value, caseSensitive));
		}
		
		// Middle name
		//MHV_CodeCR1387 - Added middle name in the query
		if (LOG.isDebugEnabled()) {
			LOG.debug("************** In PatientInformationQueryDaoImpl.getQuery() calling criteria.getMiddleName(): " + criteria.getMiddleName());
		}

		if (!StringUtils.isBlank(criteria.getMiddleName())) {
			String value = criteria.getMiddleName();
			terms.add(getPossiblyWildcardedTerm("patient.middleName", "middleName", 
                value, caseSensitive));
			if (LOG.isDebugEnabled()) {
				LOG.debug("************** In PatientInformationQueryDaoImpl.getQuery() calling getValue(value, caseSensitive): " + getValue(value, caseSensitive));
			}
			values.put("middleName", getValue(value, caseSensitive));
		}
		
		// Last name
		if (!StringUtils.isBlank(criteria.getLastName())) {
			String value = criteria.getLastName();
			terms.add(getPossiblyWildcardedTerm("patient.lastName", "lastName", 
                value, caseSensitive));
			values.put("lastName", getValue(value, caseSensitive));
		}
		
		// Email
		if (!StringUtils.isBlank(criteria.getEmail())) {
			String value = criteria.getEmail();
			terms.add(getPossiblyWildcardedTerm("patient.email", "email", 
                value, caseSensitive));
			values.put("email", getValue(value, caseSensitive));
		}
		
		// MHV ID
		if (!StringUtils.isBlank(criteria.getMhvId())) {
			String value = criteria.getMhvId();
			terms.add(getPossiblyWildcardedTerm("patient.mhvId", "mhvId", 
                value, caseSensitive));
			values.put("mhvId", getValue(value, caseSensitive));
		}
		
		// SSN
		if (!StringUtils.isBlank(criteria.getSsn())) {
			terms.add(getEqualsTerm("patient.ssn", "ssn"));
			values.put("ssn", criteria.getSsn());
		}
		
		// username
		if (!StringUtils.isBlank(criteria.getUsername())) {
			String value = criteria.getUsername();
			terms.add(getPossiblyWildcardedTerm("patient.username", "username", 
                value, caseSensitive));
			values.put("username", getValue(value, caseSensitive));
		}
		
		// Date of Birth
		if (criteria.getDateOfBirth() != null) {
			Object value = criteria.getDateOfBirth();
			terms.add(getEqualsTerm("patient.birthDate", "birthDate"));
			values.put("birthDate", value);
		}
		
		// Authentication Date (From)
		if (criteria.getFromAuthenticationDate() != null) {
			Object from = criteria.getFromAuthenticationDate();
			terms.add(getGreaterThanOrEqualsTerm("patient.authenticationDate", 
                "fromAuthenticationDate"));
			values.put("fromAuthenticationDate", from);
			criteria.setStatusAuthenticated(Boolean.TRUE);
			criteria.setStatusPendingUnauthentication(Boolean.TRUE);

		}
		
		//Authentication Date (To)
		if (criteria.getToAuthenticationDate() != null) {
			Object from = criteria.getToAuthenticationDate();
			terms.add(getLessThanOrEqualsTerm("patient.authenticationDate", 
                "toAuthenticationDate"));
			values.put("toAuthenticationDate", from);
			criteria.setStatusAuthenticated(Boolean.TRUE);
			criteria.setStatusPendingUnauthentication(Boolean.TRUE);
		}
		
		//JAZZ TASK#20524 - Matched/Unmatched check boxes Implementation
		if (criteria.getStatusMatched() && !criteria.getStatusUnmatched()) {
			terms.add("patient.matchedDate is not null");
		} else if (!criteria.getStatusMatched() && criteria.getStatusUnmatched()) {
			terms.add("patient.matchedDate is null");
		}
		
		//MHV_CodeCR1514 - US12.4 MVI Compliance Implementation - Added Correlation Status to the query -->
		if (criteria.getStatusCorrelated()
            || criteria.getStatusPendingCorrelation()
            || criteria.getStatusPendingUncorrelated()
            || criteria.getStatusUncorrelated() 
            || criteria.getStatusFailedCorrelation()
            || criteria.getStatusFailedUncorrelation() 
            || criteria.getStatusCorrelationInvalid())
        {
			terms.add(getCorrelationInTerm(criteria));
		}
		
		// Authentication Status
		if ((criteria.getStatusAuthenticated()) 
            || criteria.getStatusInProcess() 
            || criteria.getStatusPendingDataMismatch()
            || criteria.getStatusPrerequisitesComplete() 
            || criteria.getStatusUnauthenticated()) 
        {
			terms.add(getInTerm(criteria));
		}
		
		// Authenticating Facility
		if (criteria.getAuthenticatingFacilityInfoId() != null) {
			Object value = criteria.getAuthenticatingFacilityInfoId();
			terms.add(getEqualsTerm("patient.authenticatingFacilityInfoId", 
                "authenticatingFacilityInfoId"));
			values.put("authenticatingFacilityInfoId", value);
		}
		
		// Treating Facility
		if (criteria.getTreatingFacilityId() != null) {
			Object value = criteria.getTreatingFacilityId();
			terms.add(getEqualsTerm("patient.treatingFacilityId", 
                "treatingFacilityId"));
			values.put("treatingFacilityId", value);
		}
		
		String qString = buildQueryString(terms);
		
		if (LOG.isDebugEnabled()) {
			LOG.debug("Raw query string: " + qString);
		}
		
		Session session = getSession(true);
		Query query = session.createQuery(qString);
		populateQuery(query, values);
		
		if (LOG.isDebugEnabled()) {
			LOG.debug("Populated query string: " + query.getQueryString());
		}
		return query;
	}
	
	//MHV_CodeCR1514 - US12.4 MVI Compliance Implementation - Added Correlation Status to the query -->
	private String getCorrelationInTerm(InPersonAuthenticationCriteria criteria) {
		StringBuffer buff = new StringBuffer(50);
		List<String> values = new ArrayList<String>();
		if (criteria.getStatusUncorrelated()) {
			values.add(new Integer(PatientCorrelationStatus.getEnum(PatientCorrelationStatus.UNCORRELATED).getValue()).toString());
			//Jazz Defect#29291 - Made sure that matched correlation status is also considered when uncorrelated status is checked.
			values.add(new Integer(PatientCorrelationStatus.getEnum(PatientCorrelationStatus.MVIDATAMATCH).getValue()).toString());
		}
        if (criteria.getStatusPendingCorrelation()) {
			values.add(new Integer(PatientCorrelationStatus.getEnum(PatientCorrelationStatus.PENDINGCORRELATION).getValue()).toString());
        }
		if (criteria.getStatusCorrelated()) {
			values.add(new Integer(PatientCorrelationStatus.getEnum(PatientCorrelationStatus.CORRELATED).getValue()).toString());
		}
        if (criteria.getStatusPendingUncorrelated()) {
			values.add(new Integer(PatientCorrelationStatus.getEnum(PatientCorrelationStatus.PENDINGUNCORRELATION).getValue()).toString());
        }
		if (criteria.getStatusCorrelationInvalid()) {
			values.add(new Integer(PatientCorrelationStatus.getEnum(PatientCorrelationStatus.CORRELATIONINVALID).getValue()).toString());
		}
		
		if (criteria.getStatusFailedCorrelation()) {
			values.add(new Integer(PatientCorrelationStatus.getEnum(PatientCorrelationStatus.FAILEDCORRELATION).getValue()).toString());
		}
		if (criteria.getStatusFailedUncorrelation()) {
			values.add(new Integer(PatientCorrelationStatus.getEnum(PatientCorrelationStatus.FAILEDUNCORRELATION).getValue()).toString());
		}
		
		buff.append("patient.correlationStatus in (");
		for (Iterator i = values.iterator(); i.hasNext();) {
			buff.append(i.next());

			if (i.hasNext()) {
				buff.append(", ");
			}
		}
		buff.append(") ");
		return buff.toString();
	}

	private String getInTerm(InPersonAuthenticationCriteria criteria) {
		StringBuffer buff = new StringBuffer(50);
		List<String> values = new ArrayList<String>();
		if (criteria.getStatusAuthenticated()) {
			values.add(AuthenticationStatus.AUTHENTICATED);
		}
		if (criteria.getStatusInProcess()) {
			values.add(AuthenticationStatus.INPROCESS);
		}
        if (criteria.getStatusPendingDataMismatch()) {
            values.add(AuthenticationStatus.PENDINGDATAMISMATCH);
        }
		if (criteria.getStatusPrerequisitesComplete()) {
			values.add(AuthenticationStatus.PREREQUISITESCOMPLETE);
		}
		if (criteria.getStatusUnauthenticated()) {
			values.add(AuthenticationStatus.UNAUTHENTICATED);
			buff.append("(");
		}
		
		buff.append("patient.authenticationStatus in (");
		for (Iterator i = values.iterator(); i.hasNext();) {
			buff.append("'");
			buff.append(i.next());
			buff.append("'");

			if (i.hasNext()) {
				buff.append(", ");
			}
		}
		buff.append(")");
		if (criteria.getStatusUnauthenticated()) {
			buff.append("or patient.authenticationStatus is null");
			buff.append(")");
		}
		return buff.toString();
	}

	private String getGreaterThanOrEqualsTerm(String propertyName, 
        String parameterName) 
    {
		StringBuffer buff = new StringBuffer(50);
		buff.append(propertyName);
		buff.append(" >= :");
		buff.append(parameterName);
		return buff.toString();
	}
	
	private String getLessThanOrEqualsTerm(String propertyName, 
        String parameterName) 
    {
		StringBuffer buff = new StringBuffer(50);
		buff.append(propertyName);
		buff.append(" <= :");
		buff.append(parameterName);
		return buff.toString();
	}

	private String getValue(String value, boolean caseSensitive) {
		if (value.indexOf("*") != -1) {
			value = value.replace('*', '%');
		}
		return caseSensitive ? value : value.toLowerCase();
	}

	private String buildQueryString(List<String> terms) {
		int termCount = 0;
		StringBuffer buff = new StringBuffer(BASE);
		
		for (String term : terms) {
			if (termCount == 0) {
				buff.append(WHERE);
			} else {
				buff.append(AND);
			}
			buff.append(term);
			termCount++;
		}
		
		return buff.toString();
	}

	private void populateQuery(Query query, Map<String, Object> values) {
		for (Iterator<Map.Entry<String, Object>> iter = values.entrySet().
                iterator(); 
             iter.hasNext();) 
        {
			Map.Entry<String, Object> i = iter.next();
			String key = i.getKey();
			Object value = i.getValue();
			query.setParameter(key, value);
		}
	}

	private String getCaseInsensitiveLikeTerm(String propertyName, 
        String parameterName) 
    {
		StringBuffer buff = new StringBuffer(50);
		buff.append("lower(");
		buff.append(propertyName);
		buff.append(") like :");
		buff.append(parameterName);
		return buff.toString();
	}
	
	private String getLikeTerm(String propertyName, String parameterName) {
		StringBuffer buff = new StringBuffer(50);
		buff.append(propertyName);
		buff.append(" like :");
		buff.append(parameterName);
		return buff.toString();
	}
	
	private String getEqualsTerm(String propertyName, String parameterName) {
		StringBuffer buff = new StringBuffer(50);
		buff.append(propertyName);
		buff.append(" = :");
		buff.append(parameterName);
		return buff.toString();
	}

	private String getCaseInsensitiveEqualsTerm(String propertyName, 
        String parameterName) 
    {
		StringBuffer buff = new StringBuffer(50);
		buff.append("lower(");
		buff.append(propertyName);
		buff.append(")");
		buff.append(" = :");
		buff.append(parameterName);
		return buff.toString();
	}
	
	private String getPossiblyWildcardedTerm(String propertyName, 
        String parameterName, String value, boolean caseSensitive) 
    {
		value = value.replace('*', '%');
		if (value.indexOf("%") == -1) {
			if(caseSensitive)
				return getEqualsTerm(propertyName, parameterName);
            return getCaseInsensitiveEqualsTerm(propertyName, parameterName);
		}
        if (caseSensitive) {
        	return getLikeTerm(propertyName, parameterName);
        }
        return getCaseInsensitiveLikeTerm(propertyName, parameterName);
	}

}
